#! /usr/bin/env python
# -*- Mode: Python -*-
# -*- coding: ascii -*-

"""
Calculates the total length of the selected edges
"""

import lwsdk, math

__author__      = "Makoto Sekiguchi"
__date__        = "Aug 17 2025"
__copyright__   = "Copyright (C) 2025 narudesign"
__version__     = "0.9"
__maintainer__  = "Makoto Sekiguchi"
__status__      = "Release"
__lwver__       = "11"

class edgeLengthSum(lwsdk.ICommandSequence):
    def __init__(self, context):
        super(edgeLengthSum, self).__init__()
        
    def fast_edge_scan(self, edge_list, edge_id):
        edge_list.append(edge_id)
        return lwsdk.EDERR_NONE
        
    def get_distance(self, p1, p2):
        return math.sqrt((p1[0] - p2[0])**2 + (p1[1] - p2[1])**2 + (p1[2] - p2[2])**2)
        
    def process(self, mod_command):
        selectedEdge = []
        dist = 0
        
        mc = mod_command
        
        meo = mc.editBegin(0, 0, lwsdk.OPSEL_DIRECT)
        
        edgeCount = meo.edgeCount(meo.state, lwsdk.OPLYR_FG, lwsdk.EDCOUNT_SELECT)
        
        if edgeCount < 1:
            meo.done(meo.state, lwsdk.EDERR_NONE, 0)
            lwsdk.LWMessageFuncs().error("Please select 1 or more edges", "")          
            return lwsdk.AFUNC_OK
        
        eo_r = meo.fastEdgeScan(meo.state, self.fast_edge_scan, (selectedEdge,), lwsdk.OPLYR_FG, 1)
        
        for n in range(len(selectedEdge)):
            po1 = meo.edgePoint1(meo.state, selectedEdge[n])
            po2 = meo.edgePoint2(meo.state, selectedEdge[n])
            
            p1 = meo.pointPos(meo.state, po1)
            p2 = meo.pointPos(meo.state, po2)
            
            dist += self.get_distance(p1, p2)
            
        meo.done(meo.state, lwsdk.EDERR_NONE, 0)
        
        dist_formatted = "{:.4f}".format(dist)
        
        ui = lwsdk.LWPanels()
        panel = ui.create("SK_EdgeLengthSum")
        panel.setw(240)
        panel.seth(120)
        
        panel.ctl_nof_edges = panel.int_ctl("Selected edge")
        panel.ctl_nof_edges.set_int(len(selectedEdge))
        panel.ctl_nof_edges.set_w(170)
        panel.ctl_nof_edges.move(10, 10)
        
        panel.input_text = panel.str_ctl("Edge length sum", 15)
        panel.input_text.set_str(dist_formatted)
        panel.input_text.set_w(170)
        panel.input_text.move(10, 40)
        
        if panel.open(lwsdk.PANF_CANCEL) == 0:
            ui.destroy(panel)
        
        return lwsdk.AFUNC_OK

ServerTagInfo = [
    ("SK_EdgeLengthSum", lwsdk.SRVTAG_USERNAME | lwsdk.LANGID_USENGLISH ),
    ("SK_EdgeLengthSum", lwsdk.SRVTAG_BUTTONNAME | lwsdk.LANGID_USENGLISH )
]
ServerRecord = {
    lwsdk.CommandSequenceFactory("SK_EdgeLengthSum", edgeLengthSum) : ServerTagInfo
}